/// <summary>
/// Nutty Software Open WebGL Framework
/// 
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// The role of the tone map shader is to compress HDR images to LDR before displaying
/// on the monitor. This shader uses the minimal version of Reinhard's tone mapping algorithm.
/// </summary>


/// <summary>
/// Constructor.
/// </summary>
function ToneMapShader ()
{
	/// <summary>
	/// Setup inherited members.
	/// </summary>
	ImageShader.call(this);
	
	
	/// <summary>
	/// Shader variables.
	/// </summary>
	this.mAvgLuminanceId;
	this.mLuminanceWhiteId;
	this.mScaleId;
	this.mToneMapAlgorithmId;
	
	
	/// <summary>
	/// Stores the average luminance used in the calculation of Reinhard's tonemap.
	/// <summary>
	this.AvgLuminance = 0.0;
	
	
	/// <summary>
	/// AKA white luminance. Stores the smallest luminance that will be mapped to pure white.
	/// Reinhard sets this value to the maximum luminance in the image.
	/// <summary>
	this.MaxLuminance = 1.0;
	
	
	/// <summary>
	/// Modifer for adjusting the scale of the luminance (also called the key of the image).
	/// Typical values are betwen 0.0 and 1.0. Should be based on the strength of the
	/// average lumiosity in the image.
	/// <summary>
	this.Scale = 0.25;
	
	
	/// <summary>
	/// Identifies the type of tone mapping algorithm to use.
	/// <summary>
	this.ToneMapAlgorithm = 1;
	

	// This shader doesn't use these matrices (from inherited ImageShader class),
	// so just initialize them to identity.
	this.Projection = new Matrix();
	this.View = new Matrix();
}


/// <summary>
/// Enumeration of the possible tone mapping algorithms.
/// <summary>
ToneMapShader.Algorithm =
{
	None : 0,
	Reinhard : 1
}


/// <summary>
/// Prototypal Inheritance.
/// </summary>
ToneMapShader.prototype = new ImageShader();
ToneMapShader.prototype.constructor = ToneMapShader;


/// <summary>
/// Implementation.
/// </summary>
ToneMapShader.prototype.Init = function ()
{
	ImageShader.prototype.Init.call(this);

	// Tone map properties
	this.mAvgLuminanceId = this.GetVariable("AvgLuminance");
	this.mMaxLuminanceId = this.GetVariable("MaxLuminance");
	this.mScaleId = this.GetVariable("Scale");
	this.mToneMapAlgorithmId = this.GetVariable("ToneMapAlgorithm");
}


/// <summary>
/// Implementation.
/// </summary>
ToneMapShader.prototype.Enable = function ()
{
	ImageShader.prototype.Enable.call(this);
	
	// Tone map settings
	this.SetVariable(this.mAvgLuminanceId, this.AvgLuminance);
	this.SetVariable(this.mMaxLuminanceId, this.MaxLuminance);
	this.SetVariable(this.mScaleId, this.Scale);
	this.SetVariableInt(this.mToneMapAlgorithmId, this.ToneMapAlgorithm);
}


/// <summary>
/// Implementation.
/// </summary>
ToneMapShader.prototype.Draw = function (entity, numPoints, numIndices)
{
	// Draw
	ImageShader.prototype.Draw.call(this, entity, numPoints, numIndices);
}