/// <summary>
/// Nutty Software Open WebGL Framework
/// 
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// The mipmap shader takes as input an FBO that needs to be mipmapped manually. This shader
/// will continue downsampling the FBO until it reaches a resolution of 1x1 pixel. It is assumed
/// the colour attachment in the FBO has been set to use mipmaps.
///
/// Due to limitations with WebGL, we cannot:
/// 1. Render to a specific mipmap level in an FBO
/// 2. Cannot set the texture base level (assuming 1. was possible)
/// 3. Cannot set the texture bias level (assuming 1. was possible)
///
/// Due to these sad limitations, we have to create mipmaps directly onto the main frame buffer.
/// </summary>


/// <summary>
/// Constructor.
/// </summary>
function MipmapShader ()
{
	/// <summary>
	/// Setup inherited members.
	/// </summary>
	ImageShader.call(this);
	
	
	/// <summary>
	/// Shader variables.
	/// </summary>
	this.mMipLevelBiasId;
	
	
	/// <summary>
	/// Gets or sets the mip level bias to use in the shader. This is used to select the correct
	/// mipmap to filter with. Using a bias is difficult to work with because the video card will
	/// do it's own LOD and adding a bias on that is unpredictable. A better option would be to
	/// have the shader set the LOD level, but this is not supported in WebGL so we have to make due.
	/// </summary>
	this.mMipLevelBias;
	
	
	/// <summary>
	/// Stores the average pixel data (r,g,b,a) generated from the mipmap.
	/// </summary>
	this.mAvgPixel = new Uint8Array(4);
}


/// <summary>
/// Prototypal Inheritance.
/// </summary>
MipmapShader.prototype = new ImageShader();
MipmapShader.prototype.constructor = MipmapShader;


/// <summary>
/// Implementation.
/// </summary>
MipmapShader.prototype.Init = function ()
{
	ImageShader.prototype.Init.call(this);
	
	// Mipmap properties
	this.mMipLevelBiasId = this.GetVariable("MipLevelBias");
}


/// <summary>
/// Implementation.
/// </summary>
MipmapShader.prototype.Enable = function ()
{
	ImageShader.prototype.Enable.call(this);
	
	// Mipmap settings
}


/// <summary>
/// Implementation.
/// </summary>
MipmapShader.prototype.Draw = function (viewport, fbo, entity, numPoints, numIndices)
{
	// Get mipmap level 0 size (ie: the size of the texture with maximum quality)
	var material = entity.ObjectMaterial;
	var viewportWidth = viewport.width;
	var viewportHeight = viewport.height;
	
	var mipLevel = 1;
	this.mMipLevelBias = -1.0;
		
	// Activate the first (and only) texture unit
	gl.activeTexture(gl.TEXTURE0);
	material.Texture[0].Bind();
	
	// Disable depth reads and writes
	gl.disable(gl.DEPTH_TEST);
	gl.depthMask(false);
	
	// Continue downsampling until we reach a 1x1 pixel.
	// Assume width and height are square
	var size = Math.floor(fbo.GetFrameWidth() / 2.0);
	while ( size >= 1.0 )
	{
		var width = size;
		var height = size;
	
		// Update size
		this.SetSize(width, height);
		this.SetVariable(this.mImageSizeId, this.mImageSize.x, this.mImageSize.y);
		this.SetVariable(this.mTexelSizeId, this.mTexelSize.x, this.mTexelSize.y);
		viewport.width = width;
		viewport.height = height;
		gl.viewport(0, 0, width, height);
		
		// Update mip level
		// OpenGL will factor in the viewport size in its LOD calculations, so we don't have to adjust
		// the mip level bias by any more than -1 (-1 to use the previous level we're mipmapping).
		this.SetVariable(this.mMipLevelBiasId, this.mMipLevelBias);
		
		// Draw to main framebuffer, then copy it to the texture mipmap level later. This is necessary
		// because WebGL does not allow you to render directly to the mipmap level.
		BaseShader.prototype.Draw.call(this, entity.ObjectEntity, numPoints, numIndices);
		
		// Copy framebuffer to the specified mipmap level in the texture. We will use this for the next
		// mipmap pass.
		gl.copyTexImage2D(gl.TEXTURE_2D, mipLevel, gl.RGBA, 0, 0, width, height, 0);
		++mipLevel;
		size = Math.floor(size / 2.0);
	}
	
	// For HDR, we need the avg/max luminance. At this point, a 1x1 rendering was made to the framebuffer.
	// Since WebGL does not support glGetTexImage(...), we need to extract the RGBA value directly from the
	// framebuffer.
	gl.readPixels(0, 0,
				  1, 1,
				  gl.RGBA,
				  gl.UNSIGNED_BYTE,
				  this.mAvgPixel);
	
	// Re-enable GL settings
	gl.enable(gl.DEPTH_TEST);
	gl.depthMask(true);
	
	// Restore canvas size
	viewport.width = viewportWidth;
	viewport.height = viewportHeight;
}


/// <summary>
/// Gets the average pixel value calculated from the lowest mip level.
/// </summary>
/// <returns>The average pixel value calculated from the lowest mip level.</returns>
MipmapShader.prototype.GetAvgPixel = function ()
{
	return this.mAvgPixel;
}