/// <summary>
/// Nutty Software Open WebGL Framework
/// 
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// This scene demonstrates how to render HDR graphics. When working with HDR, the lighting portion
/// of the render pipeline needs to support floating point values. This affects lightmaps, shaders,
/// and frame buffer objects.
///
/// Note that while you will see I use the term "floating point" throughout, in order to maximize compatibility
/// with older hardware (and indeed be OpenGL ES 2.0 compatible), this demo will demonstrate HDR lighting
/// using the Radiance RGBE format. RGBE is a 32bit image format that extends the dynamic range of a 24bit RGB
/// image by incorporating a shared exponent (the E in RGBE). This gives the image some of the advantages of
/// floating point values while still supporting older hardware and specifications. There are however some limitations.
///
/// RGBE Pros:
/// 1. Store HDR lighting in a standard 32bit RGBA texture space.
/// 2. Smaller memory footprint compared to the equivalent floating point formats.
///
/// RGBE Cons:
/// 1. Has some artifacts such as banding and rings due to precision loss with integer arithmatic.
/// 2. Does not support hardware texture filtering or mipmapping.
///
/// Steps
/// 1. Render the scene with normal transform and lighting to a floating point colour buffer (texture).
///    Apply floating point lightmaps as needed.
///    Do not clamp output values. Pixels should be able to store intensities beyond 1.0.
///
/// 2. Find the average and maximum luminance of the scene. The easiest way to do this is to generate
///    a mipmap of the rendered texture all the way down to 1x1. That pixel will represent the average
///    or maximum luminance in the scene, which you suply to the tone map shader.
///
/// 3. Pass the rendered texture into a tone mapping shader. The tone mapping shader is reponsible for
///    compressing the HDR texture into an LDR texture.
///
/// 4. Finally, apply gamma correction.
/// </summary>


/// <summary>
/// Constructor.
/// </summary>
function HdrScene ()
{
	/// <summary>
	/// Setup inherited members.
	/// </summary>
	BaseScene.call(this);
	
	
	/// <summary>
	/// Gets or sets the dimensions of the FBO, which may or may not be the same size
	/// as the window.
	/// </summary>
	this.mFboDimension = null;
	
	
	/// <summary>
	/// This framebuffer object stores the HDR rendered scene to an RGB texture. Normally this would
	/// be a floating point texture, but this demo will use the Radiance RGBE format instead.
	/// </summary>
	this.mFboScene = null;
	this.mFboSceneColourBuffer = null;
	this.mFboSceneDepthBuffer = null;
	
	
	/// <summary>
	/// This framebuffer object stores the log-luminance values of the HDR texture. This is a prelude to
	/// the mipmapping stage where we will find the average and maximum luminance values.
	/// </summary>
	this.mFboLogLuminance = null;
	this.mFboLogLuminanceColourBuffer = null;
	
	
	/// <summary>
	/// The HDR shader responsible for rendering the scene to a floating point texture.
	/// </summary>
	this.mHdrShader = null;
	
	
	/// <summary>
	/// Shader for compressing HDR to an LDR texture.
	/// </summary>
	this.mToneMapShader = null;
	
	
	/// <summary>
	/// Shader for creating mipmaps for RGBE textures since the hardware can't do this automatically
	/// for us. These shaders will find the average and maximum luminance values used in Reinhard's
	/// tone mapping algorithm.
	/// </summary>
	this.mMipmapShaderAvg = null;
	this.mMipmapShaderMax = null;
	
	
	/// <summary>
	/// Shader to convert an RGBE image to log luminance.
	/// </summary>
	this.mLogLuminanceShader = null;
	
	
	/// <summary>
	/// Surface (rectangle) to perform post-processing effects.
	/// It is designed to fit the size of the viewport.
	/// </summary>
	this.mSurface = null;
	
	
	/// <summary>
	/// Stores the textures used by this scene.
	/// </summary>
	this.mTexture = new Array();
	
	
	/// <summary>
	/// Stores a reference to the canvas DOM element, which is used to reset the viewport
	/// back to its original size after rendering to the FBO, which may use a different
	/// dimension.
	/// </summary>
	this.mCanvas = null;
	
	
	/// <summary>
	/// Stores the average luminance of the current frame. On each subsequent render,
	/// the tone map shader will slowly be incremented to reach this value. This prevents sudden
	/// alterations in the image brightness when viewing different luminance levels.
	/// </summary>
	this.mAvgLuminance = 0;
	
	
	/// <summary>
	/// Stores the maximum luminance of the current frame. On each subsequent render,
	/// the tone map shader will slowly be incremented to reach this value. This prevents sudden
	/// alterations in the image when viewing different luminance levels.
	/// </summary>
	this.mMaxLuminance = 0;
	
	
	/// <summary>
	/// Mouse controls for camera movement.
	/// </summary>
	this.mMouseDown = false;
	this.mMouseStartPos = new Point();
	this.mCameraHeight = 1.0;
	this.mCameraStartPos = new Point(0.0, this.mCameraHeight, 0.0);
	
	
	/// <summary>
	/// UI members.
	/// </summary>
	this.mDivLoading = null;
	this.mTxtLoadingProgress = null;
	this.mControls = null;
	this.mCboxToneMapAlgorithm = null;
	this.mLblAvgLuminance = null;
	this.mLblMaxLuminance = null;
	this.mLblAutoAvgLuminance = null;
	this.mLblAutoMaxLuminance = null;

	this.mNumResources = 0;
}


/// <summary>
/// Prototypal Inheritance.
/// </summary>
HdrScene.prototype = new BaseScene();
HdrScene.prototype.constructor = HdrScene;


/// <summary>
/// Called when the mouse button is pressed.
/// </summary>
HdrScene.prototype.OnMouseDown = function (key)
{
	// Record the current mouse position
	var owner = this.Owner;
	owner.mMouseDown = true;
	owner.mMouseStartPos.x = key.pageX - this.offsetLeft;
	owner.mMouseStartPos.y = key.pageY - this.offsetTop;
}


/// <summary>
/// Called when there is mouse movement. Adjust the camera viewing matrix.
/// </summary>
HdrScene.prototype.OnMouseMove = function (key)
{
	// Process mouse movement only when the mouse button is held down
	var owner = this.Owner;
	if ( owner.mMouseDown )
	{
		// Calculate delta position
		var x = (owner.mMouseStartPos.x - (key.pageX - this.offsetLeft)) * 0.5;
		var y = (owner.mMouseStartPos.y - (key.pageY - this.offsetTop)) * 0.1;
		
		// Reset position for next movement
		owner.mMouseStartPos.x = key.pageX - this.offsetLeft;
		owner.mMouseStartPos.y = key.pageY - this.offsetTop;
		
		// Update view matrix. Y-axis translates, X-axis rotates
		var matrix = owner.mViewMatrix;
		var position = matrix.GetTranslation();
		var rotation = matrix.GetRotation();
		matrix.Rotate(0.0, rotation.y + x, 0.0);
		
		var xDir = new Point(matrix.MMatrix[0], matrix.MMatrix[1], matrix.MMatrix[2]);
		var zDir = new Point(matrix.MMatrix[8], matrix.MMatrix[9], matrix.MMatrix[10]);
		
		matrix.Translate(position.x + zDir.x * y,
						 owner.mCameraHeight,
						 position.z + zDir.z * y);
		owner.mInvViewMatrix = matrix.Inverse();
		owner.mHdrShader.View = owner.mInvViewMatrix;
		
		// Position the sky box with the camera
		position = matrix.GetTranslation();
		owner.mEntity[owner.mEntity.length - 1].ObjectMatrix.Translate(position.x, position.y, position.z);
	}
}


/// <summary>
/// Called when the mouse button is released.
/// </summary>
HdrScene.prototype.OnMouseUp = function (key)
{
	var owner = this.Owner;
	owner.mMouseDown = false;
}


/// <summary>
/// Implementation.
/// </summary>
HdrScene.prototype.Start = function ()
{
	// Setup members and default values
	this.mShader = new Array();
	this.mCanvas = document.getElementById("Canvas");
	
	
	// Specify the FBO dimensions
	// For mipmapping purposes, keep this to a power of 2 rather than use the canvas size
	//this.mFboDimension = new Point(this.mCanvas.width, this.mCanvas.height);
	this.mFboDimension = new Point(1024, 1024);
	
	
	// Construct FBO for rendering the HDR scene to an RGBE (Radiance) texture
	this.mFboSceneColourBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Colour);
	this.mFboSceneColourBuffer.TextureObject = new GLTexture2D();
	this.mFboSceneColourBuffer.TextureObject.Create(this.mFboDimension.x, this.mFboDimension.y, Texture.Format.Rgba, SamplerState.PointClamp);
	
	this.mFboSceneDepthBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Depth);
	this.mFboSceneDepthBuffer.RenderBufferFormat = Texture.Format.Depth16;
	
	this.mFboScene = new GLFrameBufferObject();
	this.mFboScene.Create(this.mFboDimension.x, this.mFboDimension.y);
	this.mFboScene.AttachBuffer(this.mFboSceneColourBuffer);
	this.mFboScene.AttachBuffer(this.mFboSceneDepthBuffer);
	
	// Setup the FBO to record the log luminance and its mipmaps
	var sampler = new SamplerState(SamplerState.PointClamp);
	sampler.HasMipmap = true;
	this.mFboLogLuminanceColourBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Colour);
	this.mFboLogLuminanceColourBuffer.TextureObject = new GLTexture2D();
	this.mFboLogLuminanceColourBuffer.TextureObject.Create(this.mFboDimension.x, this.mFboDimension.y, Texture.Format.Rgba, sampler);
	
	this.mFboLogLuminance = new GLFrameBufferObject();
	this.mFboLogLuminance.Create(this.mFboDimension.x, this.mFboDimension.y);
	this.mFboLogLuminance.AttachBuffer(this.mFboLogLuminanceColourBuffer);
	
	
	// Construct the surface used to for post-processing images
	var rectMesh = new Rectangle(1.0, 1.0);
	var rectVbo = new GLVertexBufferObject();
	rectVbo.Create(rectMesh);
	this.mSurface = new Entity();
	this.mSurface.ObjectEntity = rectVbo;
	this.mSurface.ObjectMatrix = new Matrix(4, 4);
	this.mSurface.ObjectMaterial.Texture = new Array();
	this.mSurface.ObjectMaterial.Texture.push(this.mFboSceneColourBuffer.TextureObject);
	
	
	// Construct the scene to be rendered
	this.mEntity = HdrSceneGen.Create();

	
	// Prepare resources to download
	
	// Basic transform and lighting shader
	this.mResource.Add(new ResourceItem("hdr.vs", null, "./shaders/hdr.vs"));
	this.mResource.Add(new ResourceItem("hdr.fs", null, "./shaders/hdr.fs"));
	this.mResource.Add(new ResourceItem("image.vs", null, "./shaders/image.vs"));
	this.mResource.Add(new ResourceItem("tonemap.fs", null, "./shaders/tonemap.fs"));
	this.mResource.Add(new ResourceItem("mipmap.fs", null, "./shaders/mipmap.fs"));
	this.mResource.Add(new ResourceItem("log_luminance.fs", null, "./shaders/log_luminance.fs"));
	
	
	// Textures used in the scene (these are in RGBE format)
	this.mResource.Add(new ResourceItem("building.png", null, "./images/building.png"));
	this.mResource.Add(new ResourceItem("ground.png", null, "./images/ground.png"));
	this.mResource.Add(new ResourceItem("sky.png", null, "./images/sky.png"));
	
	// Number of resource to load (including time taken to compile and load shader programs)
	// 7 shaders to compile
	// 4 shader programs to load
	this.mNumResources = this.mResource.Item.length + 7 + 4;
	
	
	// Listen for mouse activity
	this.mCanvas.Owner = this;
	this.mCanvas.onmousedown = this.OnMouseDown;
	this.mCanvas.onmouseup = this.OnMouseUp;
	this.mCanvas.onmousemove = this.OnMouseMove;
	
	
	// Setup user interface
	this.mDivLoading = $("#DivLoading");
	this.mTxtLoadingProgress = $("#TxtLoadingProgress");
	
	// Sliders
	// Array format: DOM ID, Step, Min, Max, Initial Value, Callback Function
	var sliders =
	[
		["#ExposureSlider", 0.1, -2.0, 2.0, 1.0, this.OnExposureValueChanged],
		["#ScaleSlider", 0.01, 0.0, 1.0, 0.25, this.OnScaleValueChanged]
	];
	
	this.mControls = new Array();
	for (var i = 0; i < sliders.length; ++i)
	{
		var item = sliders[i];
	
		var slider = $(item[0]);
		slider.slider(
		{
			animate: true,
			orientation: "horizontal",
			step: item[1],
			min: item[2],
			max: item[3],
			value: item[4]
		});
		slider.on("slide", {owner : this}, item[5]);
		slider.on("slidechange", {owner : this}, item[5]);
		var sliderTxt = $(item[0] + "Txt");
		sliderTxt.text(slider.slider("value"));
		
		this.mControls.push(slider);
		this.mControls.push(sliderTxt);
	}
	
	// Tone map algorithm combobox
	this.mCboxToneMapAlgorithm = $("#CboxToneMap");
	this.mCboxToneMapAlgorithm.on("change", this, this.OnToneMapAlgorithmChanged);
	
	// UI labels
	this.mLblAvgLuminance = $("#LblAvgLuminance");
	this.mLblMaxLuminance = $("#LblMaxLuminance");
	this.mLblAutoAvgLuminance = $("#LblAutoAvgLuminance");
	this.mLblAutoMaxLuminance = $("#LblAutoMaxLuminance");
	
	
	// Start downloading resources
	BaseScene.prototype.Start.call(this);
}


/// <summary>
/// Method called when the exposure slider has changed. This controls
/// the brightness of the scene. You an see actual exposure results by
/// setting the tone mapping algorithm to none.
/// </summary>
HdrScene.prototype.OnExposureValueChanged = function (event, ui)
{
	event.data.owner.mHdrShader.Exposure = Math.pow(2, ui.value);
	event.data.owner.mToneMapShader.Exposure = ui.value;
	event.data.owner.mControls[1].text(ui.value);
}


/// <summary>
/// Method called when the key slider has changed. This is used
/// in equation 3 of Reinhard's tone mapping algorithm. It adjusts
/// the pixel's final brightness.
/// </summary>
HdrScene.prototype.OnScaleValueChanged = function (event, ui)
{
	event.data.owner.mToneMapShader.Scale = ui.value;
	event.data.owner.mControls[3].text(ui.value);
}


/// <summary>
/// Method called when the tone map algorithm has changed.
/// </summary>
HdrScene.prototype.OnToneMapAlgorithmChanged = function (event)
{
	var selected = event.currentTarget.selectedIndex;
	event.data.mToneMapShader.ToneMapAlgorithm = selected;
}


/// <summary>
/// Converts an RGBE pixel to RGB and returns the floating point value in the red channel.
/// This method is required to extract the average and maximum luminance values from the
/// generated mipmaps.
/// </summary>
HdrScene.prototype.RGBEToFloat = function (rgbe)
{
	var avgPixel = [rgbe[0], rgbe[1], rgbe[2], rgbe[3]];
	var avgLuminance = 0;
	
	if ( avgPixel[3] > 0.0 )
	{
		var value = Math.pow(2.0, avgPixel[3] - (128.0 + 8.0)); // JavaScript does not support ldexp
		for (var i = 0; i < 3; ++i)
			avgPixel[i] *= value;
			
		// Convert from log luminance to luminance and apply the delta
		var logAvgLuminance = avgPixel[0];
		avgLuminance = Math.exp(logAvgLuminance) - 1.0;
	}
	
	return avgLuminance;
}


/// <summary>
/// Implementation.
/// </summary>
HdrScene.prototype.Update = function ()
{
	BaseScene.prototype.Update.call(this);
	
	// Draw only when all resources have been loaded
	if ( this.mLoadComplete )
	{
		//
		// Step 1: Render the scene to an HDR texture
		//
		this.mHdrShader.Enable();
			this.mFboScene.Enable();
				// Set and clear viewport
				gl.viewport(0, 0, this.mFboScene.GetFrameWidth(), this.mFboScene.GetFrameHeight());
				gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
				
				// Render
				for (var i = 0; i < this.mEntity.length; ++i)
					this.mHdrShader.Draw(this.mEntity[i]);
			this.mFboScene.Disable();
		this.mHdrShader.Disable();
		
		
		//
		// Step 2: Reinhard tonemapping requires the average and maximum luminance in order
		//		   to properly compress the HDR image. So convert the RGBE image rendered
		//		   in step 1 into a log luminance texture.
		this.mLogLuminanceShader.Enable();
			this.mFboLogLuminance.Enable();
				// Render
				this.mLogLuminanceShader.Draw(this.mSurface);
			this.mFboLogLuminance.Disable();
		this.mLogLuminanceShader.Disable();
		
		
		//
		// Step 3: Generate mipmaps for the luminance texture. The smallest
		//		   mipmap level, 1x1, will contain our average luminance that
		//		   will be used in the Reinhard tone mapping algorithm.
		this.mMipmapShaderAvg.Enable();
			// Find the average luminance
			this.mSurface.ObjectMaterial.Texture[0] = this.mFboLogLuminanceColourBuffer.TextureObject;
			this.mMipmapShaderAvg.Draw(this.mCanvas, this.mFboLogLuminance, this.mSurface);
		this.mMipmapShaderAvg.Disable();

		
		// Update the displayed average luminance in the UI
		if ( this.mLblAvgLuminance != null )
		{
			// Convert the mipmap average value from RGBE to floating point luminance
			var avgLuminance = this.RGBEToFloat(this.mMipmapShaderAvg.GetAvgPixel());
			this.mLblAvgLuminance.text(avgLuminance.toFixed(3));
			
			// Update the current average luminance
			this.mAvgLuminance = avgLuminance;
			
			// Initialize the tonemap shader as well, for the first frame render
			if ( this.mToneMapShader.AvgLuminance == -1 )
				this.mToneMapShader.AvgLuminance = avgLuminance;
		}
		
		
		//
		// Step 4: Same as step 3 except we're finding the maximum luminance value.
		//
		this.mMipmapShaderMax.Enable();
			this.mMipmapShaderMax.Draw(this.mCanvas, this.mFboLogLuminance, this.mSurface);
			
			// Restore surface texture
			this.mSurface.ObjectMaterial.Texture[0] = this.mFboSceneColourBuffer.TextureObject;
		this.mMipmapShaderMax.Disable();
		
		// Update the displayed max luminance in the UI
		if ( this.mLblMaxLuminance != null )
		{
			// Convert the mipmap maximum value from RGBE to floating point luminance
			var maxLuminance = this.RGBEToFloat(this.mMipmapShaderMax.GetAvgPixel());
			this.mLblMaxLuminance.text(maxLuminance.toFixed(3));
			
			// Update the current max luminance
			this.mMaxLuminance = maxLuminance;
		}
		
		
		//
		// Step 5: Apply the tone map and gamma correct the output.
		//
		this.mToneMapShader.Enable();
			// Set and clear viewport
			gl.viewport(0, 0, this.mCanvas.width, this.mCanvas.height);
			gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
			
			// Increment the average luminance change so the changes take effect over time
			// rather than suddenly. Base the stepping size on the amount of distance that
			// needs to be covered. We don't want to wait too long.
			if ( this.mToneMapShader.ToneMapAlgorithm == 1 )
			{
				// Slowly move towards the new average luminance
				var delta = Math.abs(this.mToneMapShader.AvgLuminance - this.mAvgLuminance);
				var step = delta * 0.025;
				if ( step < 0.01 )
					step = 0.01;
				if ( delta <= step )
					this.mToneMapShader.AvgLuminance = this.mAvgLuminance;
				else if ( this.mToneMapShader.AvgLuminance < this.mAvgLuminance )
					this.mToneMapShader.AvgLuminance += step;
				else
					this.mToneMapShader.AvgLuminance -= step;
					
				// Update the label
				if ( this.mLblAutoAvgLuminance != null )
					this.mLblAutoAvgLuminance.text(this.mToneMapShader.AvgLuminance.toFixed(3));
					
				// Slowly move towards the new max luminance
				delta = Math.abs(this.mToneMapShader.MaxLuminance - this.mMaxLuminance);
				step = delta * 0.025;
				if ( step < 0.01 )
					step = 0.01;
				if ( delta <= step )
					this.mToneMapShader.MaxLuminance = this.mMaxLuminance;
				else if ( this.mToneMapShader.MaxLuminance < this.mMaxLuminance )
					this.mToneMapShader.MaxLuminance += step;
				else
					this.mToneMapShader.MaxLuminance -= step;
					
				// Update the label
				if ( this.mLblAutoMaxLuminance != null )
					this.mLblAutoMaxLuminance.text(this.mToneMapShader.MaxLuminance.toFixed(3));
			}
			
			// Render
			this.mToneMapShader.Draw(this.mSurface);
		this.mToneMapShader.Disable();
	}
}


/// <summary>
/// Implementation.
/// </summary>
HdrScene.prototype.End = function ()
{
	BaseScene.prototype.End.call(this);

	// Release FBOs
	if ( this.mFboScene != null )
	{
		this.mFboScene.Release();
		this.mFboSceneColourBuffer.TextureObject.Release();
	}
	
	if ( this.mFboLogLuminance != null )
	{
		this.mFboLogLuminance.Release();
		this.mFboLogLuminanceColourBuffer.TextureObject.Release();
	}
	
	// Release VBOs
	if ( this.mSurface != null )
		this.mSurface.ObjectEntity.Release();
		
	// Release textures
	if ( this.mTexture != null )
	{
		for (var i = 0; i < this.mTexture.length; ++i)
			this.mTexture[i].Release();
	}
	
	// Release controls
	this.mDivLoading = null;
	this.mTxtLoadingProgress = null;
	this.mControls = null;
}


/// <summary>
/// This method will update the displayed loading progress.
/// </summary>
/// <param name="increment">
/// Set to true to increment the resource counter by one. Normally hanadled by BaseScene.
/// </param>
HdrScene.prototype.UpdateProgress = function (increment)
{
	if ( increment != null )
		++this.mResourceCount;
		
	if ( this.mTxtLoadingProgress != null )
		this.mTxtLoadingProgress.html(((this.mResourceCount / this.mNumResources) * 100.0).toFixed(2));
}


/// <summary>
/// Implementation.
/// </summary>
HdrScene.prototype.OnItemLoaded = function (sender, response)
{
	BaseScene.prototype.OnItemLoaded.call(this, sender, response);
	this.UpdateProgress();
}


/// <summary>
/// This method is called to compile a bunch of shaders. The browser will be
/// blocked while the GPU compiles, so we need to give the browser a chance
/// to refresh its view and take user input while this happens (good ui practice).
/// </summary>
HdrScene.prototype.CompileShader = function (shaderItem)
{
	// Compile shader
	var shader = new GLShader();
	if ( !shader.Create((shaderItem.Name.lastIndexOf(".vs") != -1) ? shader.ShaderType.Vertex : shader.ShaderType.Fragment, shaderItem.Item) )
	{
		// Report error
		var log = shader.GetLog();
		alert("Error compiling " + shaderItem.Name + ".\n\n" + log);
		return;
	}
	else
		this.mShader.push(shader);
}


/// <summary>
/// This method is called to compile a bunch of shaders. The browser will be
/// blocked while the GPU compiles, so we need to give the browser a chance
/// to refresh its view and take user input while this happens (good ui practice).
/// </summary>
HdrScene.prototype.CompileShaders = function (index, list)
{
	var shaderItem = list[index];
	if ( shaderItem != null )
	{
		// The mipmap shader requires a define statement to be changed. Handle that special case here.
		var shaderSource = null;
		if ( shaderItem.Name == "mipmap.fs" )
		{
			// Make a copy of the original shader source and set the flag to find the average.
			shaderSource = shaderItem.Item;
			shaderItem.Item = shaderSource.replace("{FIND_TYPE}", "AVERAGE");
		}
	
		this.CompileShader(shaderItem);
		
		// Run the compiler a second time to get the mipmap generator for finding the maximum value
		if ( shaderItem.Name == "mipmap.fs" )
		{
			shaderItem.Item = shaderSource.replace("{FIND_TYPE}", "MAXIMUM");
			this.CompileShader(shaderItem);
		}
	}
	else
	{
		// Resources missing?
		alert("Missing resources");
		return;
	}
	
	// Update loading progress
	++index;
	this.UpdateProgress(true);

	if ( index < list.length )
	{
		// Move on to next shader
		var parent = this;
		setTimeout(function () { parent.CompileShaders(index, list) }, 1);
	}
	else if ( index == list.length )
	{
		// Now start loading in the shaders
		this.LoadShaders(0);
	}
}


/// <summary>
/// This method is called to load a bunch of shaders. The browser will be
/// blocked while the GPU loads, so we need to give the browser a chance
/// to refresh its view and take user input while this happens (good ui practice).
/// </summary>
HdrScene.prototype.LoadShaders = function (index, blendModes)
{
	if ( index == 0 )
	{
		// Create HDR program
		this.mHdrShader = new HdrShader();
		this.mHdrShader.Projection = this.mProjectionMatrix;
		this.mHdrShader.View = this.mInvViewMatrix;
		this.mHdrShader.Create();
		this.mHdrShader.AddShader(this.mShader[0]);	// hdr.vs
		this.mHdrShader.AddShader(this.mShader[1]);	// hdr.fs
		this.mHdrShader.Link();
		this.mHdrShader.Init();
		
		this.mShader.push(this.mHdrShader);
	}
	else if ( index == 1 )
	{
		// Create tonemap program
		this.mToneMapShader = new ToneMapShader();
		this.mToneMapShader.Create();
		this.mToneMapShader.AddShader(this.mShader[2]);	// image.vs
		this.mToneMapShader.AddShader(this.mShader[3]);	// tonemap.fs
		this.mToneMapShader.Link();
		this.mToneMapShader.Init();
		this.mToneMapShader.SetSize(1024.0, 1024.0);
		
		this.mShader.push(this.mToneMapShader);
	}
	else if ( index == 2 )
	{
		// Create mipmap program to find the average
		this.mMipmapShaderAvg = new MipmapShader();
		this.mMipmapShaderAvg.Create();
		this.mMipmapShaderAvg.AddShader(this.mShader[2]);	// image.vs
		this.mMipmapShaderAvg.AddShader(this.mShader[4]);	// mipmap.fs /w AVERAGE
		this.mMipmapShaderAvg.Link();
		this.mMipmapShaderAvg.Init();
		
		this.mShader.push(this.mMipmapShaderAvg);
		
		// Create mipmap program to find the maximum
		this.mMipmapShaderMax = new MipmapShader();
		this.mMipmapShaderMax.Create();
		this.mMipmapShaderMax.AddShader(this.mShader[2]);	// image.vs
		this.mMipmapShaderMax.AddShader(this.mShader[5]);	// mipmap.fs /w MAXIMUM
		this.mMipmapShaderMax.Link();
		this.mMipmapShaderMax.Init();
		
		this.mShader.push(this.mMipmapShaderMax);
	}
	else if ( index == 3 )
	{
		// Create log luminance program
		this.mLogLuminanceShader = new ImageShader();
		this.mLogLuminanceShader.Create();
		this.mLogLuminanceShader.AddShader(this.mShader[2]);	// image.vs
		this.mLogLuminanceShader.AddShader(this.mShader[6]);	// log_luminance.fs
		this.mLogLuminanceShader.Link();
		this.mLogLuminanceShader.Init();
		
		// Set defaults (these matrices are not used)
		this.mLogLuminanceShader.Projection = new Matrix();
		this.mLogLuminanceShader.View = new Matrix();
		
		this.mShader.push(this.mLogLuminanceShader);
		
		// Hide loading screen
		if ( this.mDivLoading != null )
			this.mDivLoading.hide();
		
		this.mLoadComplete = true;
	}
	
	
	if ( !this.mLoadComplete )
	{
		// Update loading progress
		++index;
		this.UpdateProgress(true);
	
		// Move on
		var parent = this;
		setTimeout(function () { parent.LoadShaders(index) }, 1);
	}
}


/// <summary>
/// Implementation.
/// </summary>
HdrScene.prototype.OnLoadComplete = function ()
{
	// Process shaders
	var shaderList =
	[
		"hdr.vs",
		"hdr.fs",
		"image.vs",
		"tonemap.fs",
		"mipmap.fs",	// find average
		//"mipmap.fs",	// find maximum - do not uncomment (it's handled elsewhere)
		"log_luminance.fs"
	];
	
	var shaderResource = new Array();
	for (var i = 0; i < shaderList.length; ++i)
	{
		var resource = this.mResource.Find(shaderList[i]);
		if ( resource == null )
		{
			alert("Missing resource file: " + shaderList[i]);
			return;
		}
		shaderResource.push(resource);
	}
	
	// Process textures
	this.mTexture = new Array();
	this.mTexture.push(this.mResource.Find("building.png"));
	this.mTexture.push(this.mResource.Find("ground.png"));
	this.mTexture.push(this.mResource.Find("sky.png"));
	
	for (var i = 0; i < this.mTexture.length; ++i)
	{
		var resource = this.mTexture[i];
		if ( resource != null )
		{
			this.mTexture[i] = new GLTexture2D();
			this.mTexture[i].Create(resource.Item.width, resource.Item.height, Texture.Format.Rgba, SamplerState.PointClamp, resource.Item);
		}
		
		// Assign texture to entity
		// Assume proper entity ordering here
		this.mEntity[i].ObjectMaterial.Texture = new Array();
		this.mEntity[i].ObjectMaterial.Texture.push(this.mTexture[i]);
	}
	
	// Setup camera matrices
	this.mProjectionMatrix = ViewMatrix.Perspective(60.0, 1.333, 0.1, 1024.0);
	this.mViewMatrix.PointAt(new Point(this.mCameraStartPos.x, this.mCameraStartPos.y, this.mCameraStartPos.z + 1.0),
							 this.mCameraStartPos);
	this.mInvViewMatrix = this.mViewMatrix.Inverse();
	
	// Compile shaders
	this.CompileShaders(0, shaderResource);
}