/// <summary>
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// The role of the tone map shader is to compress HDR images to LDR before displaying
/// on the monitor. This shader uses the minimal version of Reinhard's tone mapping algorithm.
/// </summary>


#ifdef GL_ES
	precision highp float;
#endif


/// <summary>
/// Uniform variables.
/// <summary>
uniform sampler2D Sample0;


/// <summary>
/// Identifies the type of tone mapping algorithm to use.
/// <summary>
uniform int ToneMapAlgorithm;


/// <summary>
/// Stores the average luminance used in the calculation of Reinhard's tonemap.
/// <summary>
uniform float AvgLuminance;


/// <summary>
/// AKA white luminance. Stores the smallest luminance that will be mapped to pure white.
/// Reinhard sets this value to the maximum luminance in the image.
/// <summary>
uniform float MaxLuminance;


/// <summary>
/// Modifer for adjusting the scale of the luminance (also called the key of the image).
/// Typical values are betwen 0.0 and 1.0. Should be based on the strength of the
/// average lumiosity in the image.
/// <summary>
uniform float Scale;


/// <summary>
/// Varying variables.
/// <summary>
varying vec2 vUv;


/// <summary>
/// ldexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float ldexp (float x, float exponent)
{
	return x * pow(2.0, exponent);
}


/// <summary>
/// Convert a 32 bit RGBE pixel to a 96 bit floating point RGB pixel.
/// <summary>
vec3 RGBEToRGB (vec4 rgbe)
{
	if ( rgbe.w > 0.0 )
	{
		rgbe *= 255.0;
		float value = ldexp(1.0, rgbe.w - (128.0 + 8.0));
		return rgbe.xyz * value;
	}
	return vec3(0.0);
}


/// <summary>
/// Gets the luminance value for a pixel.
/// <summary>
float GetLuminance (vec3 rgb)
{
	// ITU-R BT.709 primaries
	return (0.2126 * rgb.x) + (0.7152 * rgb.y) + (0.0722 * rgb.z);
}


/// <summary>
/// Convert an sRGB pixel into a CIE xyY (xy = chroma, Y = luminance).
/// <summary>
vec3 RGB2xyY (vec3 rgb)
{
	const mat3 RGB2XYZ = mat3(0.4124, 0.3576, 0.1805,
							  0.2126, 0.7152, 0.0722,
							  0.0193, 0.1192, 0.9505);
	vec3 XYZ = RGB2XYZ * rgb;
	
	// XYZ to xyY
	return vec3(XYZ.x / (XYZ.x + XYZ.y + XYZ.z),
				XYZ.y / (XYZ.x + XYZ.y + XYZ.z),
				XYZ.y);
}


/// <summary>
/// Convert a CIE xyY value into sRGB.
/// <summary>
vec3 xyY2RGB (vec3 xyY)
{
	// xyY to XYZ
	vec3 XYZ = vec3((xyY.z / xyY.y) * xyY.x,
					xyY.z,
					(xyY.z / xyY.y) * (1.0 - xyY.x - xyY.y));

	const mat3 XYZ2RGB = mat3(3.2406, -1.5372, -0.4986,
                              -0.9689, 1.8758, 0.0415, 
                              0.0557, -0.2040, 1.0570);
	
	return XYZ2RGB * XYZ;
}


/// <summary>
/// Fragment shader entry.
/// <summary>
void main ()
{
	// Convert from RGBE to RGB and get the luminance value
	vec3 rgb = RGBEToRGB(texture2D(Sample0, vUv));
	float luminance = GetLuminance(rgb);

	// Apply a tone mapping algorithm.
	// No tonemap
	if ( ToneMapAlgorithm == 0 )
	{
		// Do nothing
	}
	// Reinhard
	else if ( ToneMapAlgorithm == 1 )
	{
		// Ld(x,y) = (L(x,y) + (1.0 + (L(x,y) / Lwhite^2))) / (1 + L(x,y))
		// L(x,y) = (Scale / AvgLuminance) * Lw(x,y)
		float Lwhite = MaxLuminance * MaxLuminance;
		float L = (Scale / AvgLuminance) * luminance;
		float Ld = (L * (1.0 + L / Lwhite)) / (1.0 + L);
		
		// Ld is in luminance space, so apply the scale factor to the xyY converted
		// values from RGB space, then convert back from xyY to RGB.
		vec3 xyY = RGB2xyY(rgb);
		xyY.z *= Ld;
		rgb = xyY2RGB(xyY);
	}
	
	// Apply gamma correction
	rgb = pow(rgb, vec3(1.0 / 2.2));
	
	// Set fragment
	gl_FragColor.xyz = rgb;
	gl_FragColor.w = 1.0;
}