/// <summary>
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// The role of the mipmap shader is to reduce a texture down to 1x1 in order to find
/// an average or maximum value. To prevent duplicate shader code, this shader uses a
/// define statement to control whether or not this mipmap shader finds the average or
/// the maximum value.
/// </summary>


#ifdef GL_ES
	precision highp float;
#endif


/// <summary>
/// This parameter controls whether or not this shader will
/// find the average value or the maximum value between pixels.
/// FIND_TYPE is AVERAGE for finding the average or MAXIMUM to find the maximum.
/// <summary>
#define {FIND_TYPE}


/// <summary>
/// Uniform variables.
/// <summary>
uniform sampler2D Sample0;
uniform vec2 ImageSize;
uniform vec2 TexelSize;
uniform float MipLevelBias;


/// <summary>
/// Varying variables.
/// <summary>
varying vec2 vUv;


/// <summary>
/// ldexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float ldexp (float x, float exponent)
{
	return x * pow(2.0, exponent);
}


/// <summary>
/// frexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float frexp (float x, out float exponent)
{
	exponent = ceil(log2(x));
	return(x * exp2(-exponent));
}


/// <summary>
/// Convert a 32 bit RGBE pixel to a 96 bit floating point RGB pixel.
/// <summary>
vec3 RGBEToRGB (vec4 rgbe)
{
	if ( rgbe.w > 0.0 )
	{
		rgbe *= 255.0;
		float value = ldexp(1.0, rgbe.w - (128.0 + 8.0));
		return rgbe.xyz * value;
	}
	return vec3(0.0);
}


/// <summary>
/// Convert a 96 bit floating point RGB pixel into a 32 bit RGBE pixel.
/// <summary>
vec4 RGBToRGBE (vec3 rgb)
{
	float value = max(max(rgb.x, rgb.y), rgb.z);

	if ( value < 0.00001 )
	{
		return vec4(0.0);
	}
	else
	{
		float exponent;
		vec4 rgbe = vec4(0.0);
		value = frexp(value, exponent) * 256.0 / value;
		rgbe.xyz = rgb.xyz * value;
		rgbe.w = exponent + 128.0;
		
		return (rgbe / 255.0);
	}
}


/// <summary>
/// Gets the luminance value for a pixel.
/// <summary>
float GetLuminance (vec3 rgb)
{
	// ITU-R BT.709 primaries
	return (0.2126 * rgb.x) + (0.7152 * rgb.y) + (0.0722 * rgb.z);
}


/// <summary>
/// Bilinearly filter the RGBE texture since this cannot be done in hardware.
/// <summary>
/// <param name="uv">UV coordinates.</param>
/// <param name="mipLevelBias">Mipmap bias level.</returns>
/// <returns>Bilinearly filtered RGB pixel.</returns>
vec3 BilinearFilter (vec2 uv, float mipLevelBias)
{
	vec2 fUv = fract(uv * ImageSize);
	uv = floor(uv * ImageSize) / ImageSize;

	vec3 tl = RGBEToRGB(texture2D(Sample0, uv, mipLevelBias));
	vec3 tr = RGBEToRGB(texture2D(Sample0, uv + vec2(TexelSize.x, 0.0), mipLevelBias));
	vec3 bl = RGBEToRGB(texture2D(Sample0, uv + vec2(0.0, TexelSize.y), mipLevelBias));
	vec3 br = RGBEToRGB(texture2D(Sample0, uv + vec2(TexelSize.x, TexelSize.y), mipLevelBias));

	vec3 a = mix(tl, tr, fUv.x);
	vec3 b = mix(bl, br, fUv.x);
	return mix(a, b, fUv.y);
}


/// <summary>
/// Returns the maximum pixel.
/// <summary>
/// <param name="uv">UV coordinates.</param>
/// <param name="mipLevelBias">Mipmap bias level.</returns>
/// <returns>Pixel with the maximum luminosity.</returns>
vec4 MaxFilter (vec2 uv, float mipLevelBias)
{
	// Readjust the UV to map on the point
	uv = floor(uv * ImageSize) / ImageSize;

	// Get rgbe values
	vec4 ptl = texture2D(Sample0, uv, mipLevelBias);
	vec4 ptr = texture2D(Sample0, uv + vec2(TexelSize.x, 0.0), mipLevelBias);
	vec4 pbl = texture2D(Sample0, uv + vec2(0.0, TexelSize.y), mipLevelBias);
	vec4 pbr = texture2D(Sample0, uv + vec2(TexelSize.x, TexelSize.y), mipLevelBias);
	
	// Convert to rgb
	vec3 tl = RGBEToRGB(ptl);
	vec3 tr = RGBEToRGB(ptr);
	vec3 bl = RGBEToRGB(pbl);
	vec3 br = RGBEToRGB(pbr);
	
	// Get luminance
	float ltl = GetLuminance(tl);
	float ltr = GetLuminance(tr);
	float lbl = GetLuminance(bl);
	float lbr = GetLuminance(br);
	
	// Compare luminance and return the brightest one
	float maxLuminance = max(max(max(ltl, ltr), lbl), lbr);
	if ( ltl == maxLuminance )
		return ptl;
	else if ( ltr == maxLuminance )
		return ptr;
	else if ( lbl == maxLuminance )
		return pbl;
	else
		return pbr;
}


/// <summary>
/// Fragment shader entry.
/// <summary>
void main ()
{
#ifdef AVERAGE
	vec3 rgb = BilinearFilter(vUv, MipLevelBias);
	gl_FragColor = RGBToRGBE(rgb);
#else
	gl_FragColor = MaxFilter(vUv, MipLevelBias);
#endif
}