/// <summary>
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// This shader converts an RGBE image to log luminance. The values are encoded
/// back into RGBE on completion.
/// </summary>


#ifdef GL_ES
	precision highp float;
#endif


/// <summary>
/// Uniform variables.
/// <summary>
uniform sampler2D Sample0;


/// <summary>
/// Varying variables.
/// <summary>
varying vec2 vUv;


/// <summary>
/// ldexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float ldexp (float x, float exponent)
{
	return x * pow(2.0, exponent);
}


/// <summary>
/// frexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float frexp (float x, out float exponent)
{
	exponent = ceil(log2(x));
	return(x * exp2(-exponent));
}


/// <summary>
/// Convert a 32 bit RGBE pixel to a 96 bit floating point RGB pixel.
/// <summary>
vec3 RGBEToRGB (vec4 rgbe)
{
	if ( rgbe.w > 0.0 )
	{
		rgbe *= 255.0;
		float value = ldexp(1.0, rgbe.w - (128.0 + 8.0));
		return rgbe.xyz * value;
	}
	return vec3(0.0);
}


/// <summary>
/// Convert a 96 bit floating point RGB pixel into a 32 bit RGBE pixel.
/// <summary>
vec4 RGBToRGBE (vec3 rgb)
{
	float value = max(max(rgb.x, rgb.y), rgb.z);

	if ( value < 0.00001 )
	{
		return vec4(0.0);
	}
	else
	{
		float exponent;
		vec4 rgbe = vec4(0.0);
		value = frexp(value, exponent) * 256.0 / value;
		rgbe.xyz = rgb.xyz * value;
		rgbe.w = exponent + 128.0;
		
		return (rgbe / 255.0);
	}
}


/// <summary>
/// Gets the luminance value for a pixel.
/// <summary>
float GetLuminance (vec3 rgb)
{
	// ITU-R BT.709 Primaries
	return (0.2126 * rgb.x) + (0.7152 * rgb.y) + (0.0722 * rgb.z);
}


/// <summary>
/// Fragment shader entry.
/// <summary>
void main ()
{
	// Add 1.0 to the log luminance because anything less than 1.0
	// outputs a negative value, which RGBE doesn't support.
	const float delta = 1.0;

	// Convert from RGBE to RGB
	vec3 rgb = RGBEToRGB(texture2D(Sample0, vUv));
	
	// Get the log luminance value of the pixel
	float luminance = GetLuminance(rgb);
	float logLuminance = log(delta + luminance);
	
	// Set fragment
	gl_FragColor = RGBToRGBE(vec3(logLuminance));
}