/// <summary>
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// The HDR shader doesn't manage lighting. It assumes lighting will be provided as a
/// pre-baked RGBE lightmap. You could optionally support dynamic lighting in your own
/// project as well.
/// </summary>


#ifdef GL_ES
	precision highp float;
#endif


/// <summary>
/// Exposure controls the brightness factor of the HDR render.
/// Values greater than 0.0 increase brightness.
/// Values less than 0.0 decrease brightness.
/// <summary>
uniform float Exposure;


/// <summary>
/// Uniform variables.
/// <summary>
uniform sampler2D Sample0;


/// <summary>
/// Varying variables.
/// <summary>
varying vec2 vUv;


/// <summary>
/// ldexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float ldexp (float x, float exponent)
{
	return x * pow(2.0, exponent);
}


/// <summary>
/// frexp is not part of OpenGL ES 2.0 specification, so it is defined here.
/// <summary>
float frexp (float x, out float exponent)
{
	exponent = ceil(log2(x));
	return(x * exp2(-exponent));
}


/// <summary>
/// Convert a 32 bit RGBE pixel to a 96 bit floating point RGB pixel.
/// <summary>
vec3 RGBEToRGB (vec4 rgbe)
{
	if ( rgbe.w > 0.0 )
	{
		rgbe *= 255.0;
		float value = ldexp(1.0, rgbe.w - (128.0 + 8.0));
		return rgbe.xyz * value;
	}
	return vec3(0.0);
}


/// <summary>
/// Convert a 96 bit floating point RGB pixel into a 32 bit RGBE pixel.
/// <summary>
vec4 RGBToRGBE (vec3 rgb)
{
	float value = max(max(rgb.x, rgb.y), rgb.z);

	if ( value < 0.00001 )
	{
		return vec4(0.0);
	}
	else
	{
		float exponent;
		vec4 rgbe = vec4(0.0);
		value = frexp(value, exponent) * 256.0 / value;
		rgbe.xyz = rgb.xyz * value;
		rgbe.w = exponent + 128.0;
		
		return (rgbe / 255.0);
	}
}


/// <summary>
/// Bilinearly filter the RGBE texture since this cannot be done in hardware.
/// <summary>
/// <param name="uv">UV coordinates.</param>
/// <returns>Bilinearly filtered RGB pixel.</returns>
vec3 BilinearFilter (vec2 uv)
{
	// RGBE textures are 1024x1024
	const float ImageSize = 1024.0;
	const vec2 TexelSize = vec2(1.0 / ImageSize);
	
	// Readjust the UV to map on the point
	vec2 fUv = fract(uv * ImageSize);
	uv = floor(uv * ImageSize) / ImageSize;

	vec3 tl = RGBEToRGB(texture2D(Sample0, uv));
	vec3 tr = RGBEToRGB(texture2D(Sample0, uv + vec2(TexelSize.x, 0.0)));
	vec3 bl = RGBEToRGB(texture2D(Sample0, uv + vec2(0.0, TexelSize.y)));
	vec3 br = RGBEToRGB(texture2D(Sample0, uv + vec2(TexelSize.x, TexelSize.y)));

	vec3 a = mix(tl, tr, fUv.x);
	vec3 b = mix(bl, br, fUv.x);
	return mix(a, b, fUv.y);
}


/// <summary>
/// Fragment shader entry.
/// <summary>
void main ()
{
	// Convert from RGBE to RGB and filter the pixel since the hardware
	// can't filter RGBE pixels.
	vec3 rgb = BilinearFilter(vUv);
	
	// Apply exposure adjustment to simulate overexposure (such as a bright sun).
	// Note that exposure is calculated as 2^exposure. To save on performance, this
	// is precalculated in JavaScript.
	rgb = rgb * Exposure;
	
	// Reencode to frame buffer as RGBE
	gl_FragColor = RGBToRGBE(rgb);
}