/// <summary>
/// Nutty Software Open WebGL Framework
/// 
/// Copyright (C) 2012 Nathaniel Meyer
/// Nutty Software, http://www.nutty.ca
/// All Rights Reserved.
/// 
/// Permission is hereby granted, free of charge, to any person obtaining a copy of
/// this software and associated documentation files (the "Software"), to deal in
/// the Software without restriction, including without limitation the rights to
/// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
/// of the Software, and to permit persons to whom the Software is furnished to do
/// so, subject to the following conditions:
///     1. The above copyright notice and this permission notice shall be included in all
///        copies or substantial portions of the Software.
///     2. Redistributions in binary or minimized form must reproduce the above copyright
///        notice and this list of conditions in the documentation and/or other materials
///        provided with the distribution.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.
/// </summary>


/// <summary>
/// This scene demonstrates how to make objects glow. The steps are as follows.
///
/// 1. Render the scene with normal transform and lighting to a texture.
///
/// 2. Rerender the scene, this time with colour writes disabled for non-glowing objects and
///    colour writes enabled for glowing objects. The depth buffer will properly cull out the
///    non-visible regions of the glowing geometry.
///
///    You may choose to disable lighting if you want to simulate glows from an emissive light source.
///    You should also decide on the glowmap resolution to use. Smaller glowmaps will give faster
///    results at the expense of quality.
///
/// 3. Blur the glowmap using a seperable blur algorithm. That is, blur in the horizontal axis
///    first and then the vertical axis.
///
/// 4. Blend the texture from step 1 with the texture in step 3 using either additive blending,
///    screen blending, or a soft light blending algorithm.
/// </summary>


/// <summary>
/// Constructor.
/// </summary>
function GlowScene ()
{
	/// <summary>
	/// Setup inherited members.
	/// </summary>
	BaseScene.call(this);
	
	
	/// <summary>
	/// RNG for randomizing the objects that glow in the scene.
	/// </summary>
	this.mRandom = new RandomGenerator();
	this.mRandom.Seed(new Date().getTime());
	
	/// <summary>
	/// This framebuffer object stores the rendered scene to an RGB texture. It will be
	/// blended with the glowmap calculated later.
	/// </summary>
	this.mFboColour = null;
	this.mFboColourColourBuffer = null;
	this.mFboColourDepthBuffer = null;
	
	
	/// <summary>
	/// This framebuffer object stores the rendered objects that will cast a glow. This texture
	/// will be blurred before it is blended with the colour buffer fbo.
	/// </summary>
	this.mFboGlow = null;
	this.mFboGlowColourBuffer = null;
	this.mFboGlowDepthBuffer = null;
	
	
	/// <summary>
	/// This framebuffer object stores the view space normal vectors to texture.
	/// It uses an RGB floating point texture to store the results.
	/// </summary>
	this.mFboBlur = null;
	this.mFboBlurColourBuffer = null;

	
	/// <summary>
	/// Gets or sets the dimensions of the glowmap / blur FBO, which may or may not
	/// be the same size as the window.
	/// </summary>
	this.mFboDimension = null;
	
	
	/// <summary>
	/// The basic shader renders the scene using standard transform and lighting.
	/// It is also responsible for rendering the glowmap.
	/// </summary>
	this.mBasicShader = null;
	
	
	/// <summary>
	/// The blur shader will perform a seperable blur algorithm on the glowmap.
	/// </summary>
	this.mBlurShader = null;
	
	
	/// <summary>
	/// The glow shader blends the glowmap with the rendered scene to generate a final result
	/// with glows.
	/// </summary>
	this.mGlowShader = null;
	
	
	/// <summary>
	/// Surface (rectangle) to perform post-processing effects.
	/// It is designed to fit the size of the viewport.
	/// </summary>
	this.mSurface = null;
	
	
	/// <summary>
	/// Stores the textures used by this scene.
	/// </summary>
	this.mTexture = new Array();
	
	
	/// <summary>
	/// Stores a reference to the canvas DOM element, which is used to reset the viewport
	/// back to its original size after rendering to the FBO, which may use a different
	/// dimension.
	/// </summary>
	this.mCanvas = null;
	
	
	/// <summary>
	/// Mouse controls for camera position and zoom.
	/// </summary>
	this.mMouseDown = false;
	this.mMouseStartPos = new Point();
	this.mCameraRot = 0.0;
	this.mCameraZoom = 8.9;
	this.mCameraHeight = 2.0;
	this.mCameraStartRot = 0.0;
	this.mCameraStartZoom = 0.0;
	this.mCameraTargetPos = new Point(0.0, 0.4, 0.0);
	
	
	/// <summary>
	/// Every X frames, randomize a new set of tiles that will glow a certain colour.
	/// </summary>
	this.mFramesUntilUpdate = 60 * 3;
	this.mCurrentFrame = this.mFramesUntilUpdate;
	
	
	/// <summary>
	/// Animate the colour of the disco ball.
	/// </summary>
	this.mPrevDiscoBallColour = new Point();
	this.mNextDiscoBallColour = new Point();
	this.mDiscoBallColourDelta = 1.0;
	
	
	/// <summary>
	/// Pulsate the glows for the ceiling rings
	/// </summary>
	this.mFramesUntilRingUpdate = 60 * 0.5;
	this.mCurrentRingFrame = this.mFramesUntilRingUpdate;
	this.mRingBurstCount = 0;
	
	
	/// <summary>
	/// UI members.
	/// </summary>
	this.mDivLoading = null;
	this.mTxtLoadingProgress = null;
	this.mControls = null;
	this.mCboxGlowResolution = null;
	this.mCboxGlowMode = null;

	this.mNumResources = 0;
}


/// <summary>
/// Prototypal Inheritance.
/// </summary>
GlowScene.prototype = new BaseScene();
GlowScene.prototype.constructor = GlowScene;


/// <summary>
/// Called when the mouse button is pressed.
/// </summary>
GlowScene.prototype.OnMouseDown = function (key)
{
	// Record the current mouse position
	var owner = this.Owner;
	owner.mMouseDown = true;
	owner.mMouseStartPos.x = key.pageX - this.offsetLeft;
	owner.mMouseStartPos.y = key.pageY - this.offsetTop;
	owner.mCameraStartRot = owner.mCameraRot;
	owner.mCameraStartZoom = owner.mCameraZoom;
}


/// <summary>
/// Called when there is mouse movement. Apply rotation and zoom
/// only when the mouse button is held down.
/// </summary>
GlowScene.prototype.OnMouseMove = function (key)
{
	// Process mouse movement only when the mouse button is held down
	var owner = this.Owner;
	if ( owner.mMouseDown )
	{
		// Rotate and zoom camera about centre
		// Calculate delta position
		var x = (owner.mMouseStartPos.x - (key.pageX - this.offsetLeft)) * 0.01;
		var y = (owner.mMouseStartPos.y - (key.pageY - this.offsetTop)) * 0.07;
		
		owner.mCameraRot = owner.mCameraStartRot + x;
		owner.mCameraZoom = owner.mCameraStartZoom + y;
		
		// Impose limits on zoom
		if ( owner.mCameraZoom < 2.0 )
			owner.mCameraZoom = 2.0;
		else if ( owner.mCameraZoom > 8.9 )
			owner.mCameraZoom = 8.9;
		
		owner.mViewMatrix.PointAt(new Point(Math.sin(owner.mCameraRot) * owner.mCameraZoom, owner.mCameraHeight, Math.cos(owner.mCameraRot) * owner.mCameraZoom),
								  owner.mCameraTargetPos);
		owner.mInvViewMatrix = owner.mViewMatrix.Inverse();
		
		owner.mBasicShader.View = owner.mInvViewMatrix;
	}
}


/// <summary>
/// Called when the mouse button is released.
/// </summary>
GlowScene.prototype.OnMouseUp = function (key)
{
	var owner = this.Owner;
	owner.mMouseDown = false;
}


/// <summary>
/// Implementation.
/// </summary>
GlowScene.prototype.Start = function ()
{
	// Setup members and default values
	this.mShader = new Array();
	this.mCanvas = document.getElementById("Canvas");
	
	
	// Specify the FBO dimensions
	this.mFboDimension = new Point(128, 128);
	
	
	// Construct FBO for rendering the scene to an RGB texture
	this.mFboColourColourBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Colour);
	this.mFboColourColourBuffer.TextureObject = new GLTexture2D();
	this.mFboColourColourBuffer.TextureObject.Create(this.mCanvas.width, this.mCanvas.height, Texture.Format.Rgb, SamplerState.LinearClamp);
	
	this.mFboColourDepthBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Depth);
	this.mFboColourDepthBuffer.RenderBufferFormat = Texture.Format.Depth16;
	
	this.mFboColour = new GLFrameBufferObject();
	this.mFboColour.Create(this.mCanvas.width, this.mCanvas.height);
	this.mFboColour.AttachBuffer(this.mFboColourColourBuffer);
	this.mFboColour.AttachBuffer(this.mFboColourDepthBuffer);
	
	
	// Construct FBO for rendering the glowmap
	this.mFboGlowColourBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Colour);
	this.mFboGlowColourBuffer.TextureObject = new GLTexture2D();
	this.mFboGlowColourBuffer.TextureObject.Create(this.mFboDimension.x, this.mFboDimension.y, Texture.Format.Rgba, SamplerState.LinearClamp);
	
	this.mFboGlowDepthBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Depth);
	this.mFboGlowDepthBuffer.RenderBufferFormat = Texture.Format.Depth16;
	
	this.mFboGlow = new GLFrameBufferObject();
	this.mFboGlow.Create(this.mFboDimension.x, this.mFboDimension.y);
	this.mFboGlow.AttachBuffer(this.mFboGlowColourBuffer);
	this.mFboGlow.AttachBuffer(this.mFboGlowDepthBuffer);
	
	
	// Construct FBO for rendering the horizontal blur (vertical blur will be copied back into glowmap FBO)
	this.mFboBlurColourBuffer = new FrameBufferObject(FrameBufferObject.BufferType.Colour);
	this.mFboBlurColourBuffer.TextureObject = new GLTexture2D();
	this.mFboBlurColourBuffer.TextureObject.Create(this.mFboDimension.x, this.mFboDimension.y, Texture.Format.Rgba, SamplerState.LinearClamp);
	
	this.mFboBlur = new GLFrameBufferObject();
	this.mFboBlur.Create(this.mFboDimension.x, this.mFboDimension.y);
	this.mFboBlur.AttachBuffer(this.mFboBlurColourBuffer);
	
	
	// Construct the surface used to for post-processing images
	var rectMesh = new Rectangle(1.0, 1.0);
	var rectVbo = new GLVertexBufferObject();
	rectVbo.Create(rectMesh);
	this.mSurface = new Entity();
	this.mSurface.ObjectEntity = rectVbo;
	this.mSurface.ObjectMatrix = new Matrix(4, 4);
	this.mSurface.ObjectMaterial.Texture = new Array();
	this.mSurface.ObjectMaterial.Texture.push(null);	// Rendered scene texture
	this.mSurface.ObjectMaterial.Texture.push(null);	// Glowmap
	
	
	// Construct the scene to be rendered
	this.mEntity = GlowSceneGen.Create();

	
	// Prepare resources to download
	
	// Basic transform and lighting shader
	this.mResource.Add(new ResourceItem("basic.vs", null, "./shaders/basic.vs"));
	this.mResource.Add(new ResourceItem("basic.fs", null, "./shaders/basic.fs"));
	
	// Post-processing shaders
	this.mResource.Add(new ResourceItem("image.vs", null, "./shaders/image.vs"));
	this.mResource.Add(new ResourceItem("blur.fs", null, "./shaders/blur.fs"));
	this.mResource.Add(new ResourceItem("glow.fs", null, "./shaders/glow.fs"));
	
	
	// Textures used in the scene
	//this.mResource.Add(new ResourceItem("xyz.png", null, "./images/xyz.png"));
	
	// Number of resource to load (including time taken to compile and load shader programs)
	// 5 shaders to compile
	// 3 shader programs to load
	this.mNumResources = this.mResource.Item.length + 5 + 3;
	
	
	// Listen for mouse activity
	this.mCanvas.Owner = this;
	this.mCanvas.onmousedown = this.OnMouseDown;
	this.mCanvas.onmouseup = this.OnMouseUp;
	this.mCanvas.onmousemove = this.OnMouseMove;
	
	
	// Setup user interface
	this.mDivLoading = $("#DivLoading");
	this.mTxtLoadingProgress = $("#TxtLoadingProgress");
	
	// Sliders
	// Array format: DOM ID, Step, Min, Max, Initial Value, Callback Function
	var sliders =
	[
		["#BlurRadiusSlider", 1.0, 2.0, 10.0, 10.0, this.OnBlurRadiusValueChanged],
		["#BlurScaleSlider", 0.5, 1.0, 5.0, 1.0, this.OnBlurScaleValueChanged],
		["#BlurStrengthSlider", 0.1, 0.0, 1.0, 0.2, this.OnBlurStrengthValueChanged]
	];
	
	this.mControls = new Array();
	for (var i = 0; i < sliders.length; ++i)
	{
		var item = sliders[i];
	
		var slider = $(item[0]);
		slider.slider(
		{
			animate: true,
			orientation: "horizontal",
			step: item[1],
			min: item[2],
			max: item[3],
			value: item[4]
		});
		slider.on("slide", {owner : this}, item[5]);
		slider.on("slidechange", {owner : this}, item[5]);
		var sliderTxt = $(item[0] + "Txt");
		sliderTxt.text(slider.slider("value"));
		
		this.mControls.push(slider);
		this.mControls.push(sliderTxt);
	}
	
	// Glow resolution
	this.mCboxGlowResolution = $("#CboxGlowResolution");
	this.mCboxGlowResolution.on("change", this, this.OnGlowResolutionChanged);
	
	// Glow mode
	this.mCboxGlowMode = $("#CboxGlowMode");
	this.mCboxGlowMode.on("change", this, this.OnGlowModeChanged);
	
	
	// Start downloading resources
	BaseScene.prototype.Start.call(this);
	
	
	// Default uses a white clear colour. Reset this to black for proper glowing / blending.
	gl.clearColor(0.0, 0.0, 0.0, 0.0);
}


/// <summary>
/// Method called when the blur radius slider has changed.
/// </summary>
GlowScene.prototype.OnBlurRadiusValueChanged = function (event, ui)
{
	event.data.owner.mBlurShader.BlurAmount = ui.value;
	event.data.owner.mControls[1].text(ui.value);
}


/// <summary>
/// Method called when the blur scale slider has changed.
/// </summary>
GlowScene.prototype.OnBlurScaleValueChanged = function (event, ui)
{
	event.data.owner.mBlurShader.BlurScale = ui.value;
	event.data.owner.mControls[3].text(ui.value);
}


/// <summary>
/// Method called when the blur strength slider has changed.
/// </summary>
GlowScene.prototype.OnBlurStrengthValueChanged = function (event, ui)
{
	event.data.owner.mBlurShader.BlurStrength = ui.value;
	event.data.owner.mControls[5].text(ui.value);
}


/// <summary>
/// Method called when the glowmap resolution combo box has changed.
/// </summary>
GlowScene.prototype.OnGlowResolutionChanged = function (event)
{
	var selected = event.currentTarget.selectedIndex;
	var resolution = Math.pow(2.0, 6 + selected);
	
	event.data.mFboDimension.x = resolution;
	event.data.mFboDimension.y = resolution;
	
	// Update FBOs
	event.data.mFboGlow.Resize(resolution, resolution);
	event.data.mFboBlur.Resize(resolution, resolution);
	
	event.data.mBlurShader.SetSize(resolution, resolution);
}


/// <summary>
/// Method called when the glowmap mode combo box has changed.
/// </summary>
GlowScene.prototype.OnGlowModeChanged = function (event)
{
	event.data.mGlowShader.BlendMode = event.currentTarget.selectedIndex;
}


/// <summary>
/// Implementation.
/// </summary>
GlowScene.prototype.Update = function ()
{
	BaseScene.prototype.Update.call(this);
	
	// Draw only when all resources have been loaded
	if ( this.mLoadComplete )
	{
		//
		// Once we reach mFramesUntilUpdate, randomize a new set of tiles that will glow
		//
		++this.mCurrentFrame;
		if ( this.mCurrentFrame >= this.mFramesUntilUpdate )
		{
			// Reset ambience for tiles only (leave the rest alone)
			var count = this.mEntity.length - 6;
			for (var i = 0; i < count; ++i)
				this.mEntity[i].ObjectMaterial.Ambient.SetPoint(0.0, 0.0, 0.0);
				
			// Randomize a new set of glowing tiles
			var glowCount = this.mRandom.Random(15, 40);
			for (var i = 0; i < glowCount; ++i)
			{
				var index = this.mRandom.Random(0, count - 1);
				this.mEntity[index].ObjectMaterial.Ambient.SetPoint(this.mRandom.RandomFloat(0.0, 1.0),
																	this.mRandom.RandomFloat(0.0, 1.0),
																	this.mRandom.RandomFloat(0.0, 1.0));
			}
		
			// Reset
			this.mCurrentFrame = 0;
			
			// Randomize when the next update will occur
			this.mFramesUntilUpdate = 60.0 * this.mRandom.RandomFloat(1.0, 4.0);
		}
		
		
		//
		// Animate the disco ball colour
		//
		this.mDiscoBallColourDelta += 0.01;
		if ( this.mDiscoBallColourDelta > 1.0 )
		{
			this.mDiscoBallColourDelta = 0.0;
			
			// Copy next colour into previous and update next colour to something else
			this.mPrevDiscoBallColour.x = this.mNextDiscoBallColour.x;
			this.mPrevDiscoBallColour.y = this.mNextDiscoBallColour.y;
			this.mPrevDiscoBallColour.z = this.mNextDiscoBallColour.z;
			
			this.mNextDiscoBallColour.x = this.mRandom.RandomFloat(0.0, 1.0);
			this.mNextDiscoBallColour.y = this.mRandom.RandomFloat(0.0, 1.0);
			this.mNextDiscoBallColour.z = this.mRandom.RandomFloat(0.0, 1.0);
		}
		var entity = this.mEntity[this.mEntity.length - 2];
		entity.ObjectMaterial.Ambient.x = this.mPrevDiscoBallColour.x + (this.mNextDiscoBallColour.x - this.mPrevDiscoBallColour.x) * this.mDiscoBallColourDelta;
		entity.ObjectMaterial.Ambient.y = this.mPrevDiscoBallColour.y + (this.mNextDiscoBallColour.y - this.mPrevDiscoBallColour.y) * this.mDiscoBallColourDelta;
		entity.ObjectMaterial.Ambient.z = this.mPrevDiscoBallColour.z + (this.mNextDiscoBallColour.z - this.mPrevDiscoBallColour.z) * this.mDiscoBallColourDelta;
		
		//
		// Pulse the rings
		//
		++this.mCurrentRingFrame;
		if ( this.mCurrentRingFrame >= this.mFramesUntilRingUpdate )
		{
			// Push the colours
			var count = this.mEntity.length;
			for (var i = (count - 3); i > (count - (4 + 2)); --i)
				this.mEntity[i].ObjectMaterial.Ambient = this.mEntity[i-1].ObjectMaterial.Ambient;
				
			/*this.mEntity[count - (4 + 2)].ObjectMaterial.Ambient = new Point(entity.ObjectMaterial.Ambient.x,
																			 entity.ObjectMaterial.Ambient.y,
																			 entity.ObjectMaterial.Ambient.z);*/
			
			this.mEntity[count - (4 + 2)].ObjectMaterial.Ambient = new Point(this.mRandom.RandomFloat(0.0, 1.0),
																			 this.mRandom.RandomFloat(0.0, 1.0),
																			 this.mRandom.RandomFloat(0.0, 1.0));

			// Reset
			this.mCurrentRingFrame = 0;
			
			// Randomize when the next update will occur
			if ( this.mRingBurstCount > 0 )
			{
				this.mFramesUntilRingUpdate = 60.0 * 0.1;
				--this.mRingBurstCount;
			}
			else
			{
				this.mFramesUntilRingUpdate = 60.0 * 0.75; //this.mRandom.RandomFloat(0.25, 0.75);
				
				// % chance to burst on next pass
				if ( this.mRandom.RandomFloat(0.0, 1.0) <= 0.05 )
					this.mRingBurstCount = 16;
			}
		}
	
	
		// Restore depth check
		gl.enable(gl.DEPTH_TEST);
		gl.depthMask(true);
	
		//
		// Step 1: Render the scene to texture using a standard T&L shader
		//
		this.mBasicShader.Enable();
			this.mFboColour.Enable();
				// Set and clear viewport
				gl.viewport(0, 0, this.mFboColour.GetFrameWidth(), this.mFboColour.GetFrameHeight());
				gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
				
				// Render
				for (var i = 0; i < this.mEntity.length; ++i)
					this.mBasicShader.Draw(this.mEntity[i]);
			this.mFboColour.Disable();
		
			//
			// Step 2: Render the glowing entities to a separate texture
			//
			this.mFboGlow.Enable();
				// Set and clear viewport
				gl.viewport(0, 0, this.mFboGlow.GetFrameWidth(), this.mFboGlow.GetFrameHeight());
				gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
				
				// Render occluding geometry first (disable colour writes)
				// For efficiency, you would render this geometry using a "dumb"
				// shader with no lighting; however for simplicity I'm reusing the
				// basic shader.
				gl.colorMask(false, false, false, false);
				for (var i = 0; i < this.mEntity.length; ++i)
				{
					var ambient = this.mEntity[i].ObjectMaterial.Ambient;
					if ( ((ambient.x == 0.0) && (ambient.y == 0.0) && (ambient.z == 0.0)) ||
						 (i == (this.mEntity.length - 1)) )
						this.mBasicShader.Draw(this.mEntity[i]);
				}
				gl.colorMask(true, true, true, true);
				
				// Render glowing geometry second (exclude walls)
				for (var i = 0; i < (this.mEntity.length - 1); ++i)
				{
					var ambient = this.mEntity[i].ObjectMaterial.Ambient;
					if ( (ambient.x != 0.0) || (ambient.y != 0.0) || (ambient.z != 0.0) )
						this.mBasicShader.Draw(this.mEntity[i]);
				}
			this.mFboGlow.Disable();
		this.mBasicShader.Disable();
		
		
		//
		// Step 3: Blur the glowmap (disable depth checks and writes)
		//
		gl.disable(gl.DEPTH_TEST);
		gl.depthMask(false);
		
		this.mBlurShader.Enable();
			this.mFboBlur.Enable();
				// Set and clear viewport
				gl.viewport(0, 0, this.mFboBlur.GetFrameWidth(), this.mFboBlur.GetFrameHeight());
				gl.clear(gl.COLOR_BUFFER_BIT);
				
				// Render horizontal blur
				this.mSurface.ObjectMaterial.Texture[0] = this.mFboGlowColourBuffer.TextureObject;
				this.mBlurShader.Orientation = 0;
				this.mBlurShader.Draw(this.mSurface);
			
			// Mix with vertical blur
			this.mFboGlow.Enable();
				// Clear viewport
				gl.clear(gl.COLOR_BUFFER_BIT);
				
				this.mSurface.ObjectMaterial.Texture[0] = this.mFboBlurColourBuffer.TextureObject;
				this.mBlurShader.Orientation = 1;
				this.mBlurShader.Draw(this.mSurface);
			this.mFboGlow.Disable();
		this.mBlurShader.Disable();
		
		
		// Restore viewport
		gl.viewport(0, 0, this.mCanvas.width, this.mCanvas.height);
		gl.clear(gl.COLOR_BUFFER_BIT);
		
		
		//
		// Step 4: Blend the glowmap with the rendered scene
		//
		this.mGlowShader.Enable();
			// Set textures
			this.mSurface.ObjectMaterial.Texture[0] = this.mFboColourColourBuffer.TextureObject;	// Rendered scene from step 1
			this.mSurface.ObjectMaterial.Texture[1] = this.mFboGlowColourBuffer.TextureObject;		// Glowmap from step 3
			
			// Render
			this.mGlowShader.Draw(this.mSurface);
		this.mGlowShader.Disable();
	}
}


/// <summary>
/// Implementation.
/// </summary>
GlowScene.prototype.End = function ()
{
	BaseScene.prototype.End.call(this);

	// Cleanup
	
	// Release FBOs
	if ( this.mFboColour != null )
	{
		this.mFboColour.Release();
		this.mFboColourColourBuffer.TextureObject.Release();
	}
	
	if ( this.mFboGlow != null )
	{
		this.mFboGlow.Release();
		this.mFboGlowColourBuffer.TextureObject.Release();
	}
	
	if ( this.mFboBlur != null )
	{
		this.mFboBlur.Release();
		this.mFboBlurColourBuffer.TextureObject.Release();
	}
	
	// Release VBOs
	if ( this.mSurface != null )
		this.mSurface.ObjectEntity.Release();
		
	// Release textures
	if ( this.mTexture != null )
	{
		for (var i = 0; i < this.mTexture.length; ++i)
			this.mTexture[i].Release();
	}
	
	// Release controls
	this.mDivLoading = null;
	this.mTxtLoadingProgress = null;
	this.mControls = null;
	this.mBtnEnableBlur = null;
	this.mCboxGlowResolution = null;
}


/// <summary>
/// This method will update the displayed loading progress.
/// </summary>
/// <param name="increment">
/// Set to true to increment the resource counter by one. Normally hanadled by BaseScene.
/// </param>
GlowScene.prototype.UpdateProgress = function (increment)
{
	if ( increment != null )
		++this.mResourceCount;
		
	if ( this.mTxtLoadingProgress != null )
		this.mTxtLoadingProgress.html(((this.mResourceCount / this.mNumResources) * 100.0).toFixed(2));
}


/// <summary>
/// Implementation.
/// </summary>
GlowScene.prototype.OnItemLoaded = function (sender, response)
{
	BaseScene.prototype.OnItemLoaded.call(this, sender, response);
	this.UpdateProgress();
}


/// <summary>
/// This method is called to compile a bunch of shaders. The browser will be
/// blocked while the GPU compiles, so we need to give the browser a chance
/// to refresh its view and take user input while this happens (good ui practice).
/// </summary>
GlowScene.prototype.CompileShaders = function (index, list)
{
	var shaderItem = list[index];
	if ( shaderItem != null )
	{
		// Compile vertex shader
		var shader = new GLShader();
		if ( !shader.Create((shaderItem.Name.lastIndexOf(".vs") != -1) ? shader.ShaderType.Vertex : shader.ShaderType.Fragment, shaderItem.Item) )
		{
			// Report error
			var log = shader.GetLog();
			alert("Error compiling " + shaderItem.Name + ".\n\n" + log);
			return;
		}
		else
			this.mShader.push(shader);	
	}
	else
	{
		// Resources missing?
		alert("Missing resources");
		return;
	}
	
	// Update loading progress
	++index;
	this.UpdateProgress(true);

	if ( index < list.length )
	{
		// Move on to next shader
		var parent = this;
		setTimeout(function () { parent.CompileShaders(index, list) }, 1);
	}
	else if ( index == list.length )
	{
		// Now start loading in the shaders
		this.LoadShaders(0);
	}
}


/// <summary>
/// This method is called to load a bunch of shaders. The browser will be
/// blocked while the GPU loads, so we need to give the browser a chance
/// to refresh its view and take user input while this happens (good ui practice).
/// </summary>
GlowScene.prototype.LoadShaders = function (index, blendModes)
{
	if ( index == 0 )
	{
		// Create basic T&L program
		this.mBasicShader = new BasicShader();
		this.mBasicShader.Projection = this.mProjectionMatrix;
		this.mBasicShader.View = this.mInvViewMatrix;
		this.mBasicShader.Create();
		this.mBasicShader.AddShader(this.mShader[0]);	// basic.vs
		this.mBasicShader.AddShader(this.mShader[1]);	// basic.fs
		this.mBasicShader.Link();
		this.mBasicShader.Init();
		
		// Add light sources
		var light1 = new Light();
		light1.LightType = Light.LightSourceType.Point;
		light1.Position = new Point(0.0, 30.0, 0.0);
		light1.Attenuation.z = 0.01;
		this.mBasicShader.LightObject.push(light1);
		
		this.mShader.push(this.mBasicShader);
	}
	else if ( index == 1 )
	{
		// Create blur program
		this.mBlurShader = new BlurShader();
		this.mBlurShader.Create();
		this.mBlurShader.AddShader(this.mShader[2]);	// image.vs
		this.mBlurShader.AddShader(this.mShader[3]);	// blur.fs
		this.mBlurShader.Link();
		this.mBlurShader.Init();
		
		// Setup program
		this.mBlurShader.Projection = new Matrix(4, 4);
		this.mBlurShader.View = new Matrix(4, 4);
		this.mBlurShader.SetSize(this.mFboDimension.x, this.mFboDimension.y);
		
		this.mShader.push(this.mBlurShader);
	}
	else if ( index == 2 )
	{
		// Create glow (blend) program
		this.mGlowShader = new GlowShader();
		this.mGlowShader.Create();
		this.mGlowShader.AddShader(this.mShader[2]);	// image.vs
		this.mGlowShader.AddShader(this.mShader[4]);	// glow.fs
		this.mGlowShader.Link();
		this.mGlowShader.Init();
		
		// Setup program
		this.mGlowShader.Projection = new Matrix(4, 4);
		this.mGlowShader.View = new Matrix(4, 4);
		
		this.mShader.push(this.mGlowShader);
		
		
		// Hide loading screen
		if ( this.mDivLoading != null )
			this.mDivLoading.hide();
		
		this.mLoadComplete = true;
	}
	
	if ( !this.mLoadComplete )
	{
		// Update loading progress
		++index;
		this.UpdateProgress(true);
	
		// Move on
		var parent = this;
		setTimeout(function () { parent.LoadShaders(index) }, 1);
	}
}


/// <summary>
/// Implementation.
/// </summary>
GlowScene.prototype.OnLoadComplete = function ()
{
	// Process shaders
	var shaderList =
	[
		"basic.vs",
		"basic.fs",
		"image.vs",
		"blur.fs",
		"glow.fs"
	];
	
	var shaderResource = new Array();
	for (var i = 0; i < shaderList.length; ++i)
	{
		var resource = this.mResource.Find(shaderList[i]);
		if ( resource == null )
		{
			alert("Missing resource file: " + shaderList[i]);
			return;
		}
		shaderResource.push(resource);
	}
	
	// Setup camera matrices
	this.mProjectionMatrix = ViewMatrix.Perspective(60.0, 1.333, 0.1, 128.0);
	this.mViewMatrix.PointAt(new Point(0.0, this.mCameraHeight, this.mCameraZoom),
							 this.mCameraTargetPos);
	this.mInvViewMatrix = this.mViewMatrix.Inverse();
	
	// Compile shaders
	this.CompileShaders(0, shaderResource);
}